/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.services.s3.internal.multipart;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.internal.multipart.GenericMultipartHelper;
import software.amazon.awssdk.services.s3.internal.multipart.KnownContentLengthAsyncRequestBodySubscriber;
import software.amazon.awssdk.services.s3.internal.multipart.MpuRequestContext;
import software.amazon.awssdk.services.s3.internal.multipart.MultipartUploadHelper;
import software.amazon.awssdk.services.s3.internal.multipart.PausableUpload;
import software.amazon.awssdk.services.s3.internal.multipart.SdkPojoConversionUtils;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.ListPartsRequest;
import software.amazon.awssdk.services.s3.model.Part;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.multipart.PauseObservable;
import software.amazon.awssdk.services.s3.multipart.S3MultipartExecutionAttribute;
import software.amazon.awssdk.services.s3.multipart.S3ResumeToken;
import software.amazon.awssdk.services.s3.paginators.ListPartsPublisher;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.Pair;

@SdkInternalApi
public final class UploadWithKnownContentLengthHelper {
    private static final Logger log = Logger.loggerFor(UploadWithKnownContentLengthHelper.class);
    private final S3AsyncClient s3AsyncClient;
    private final long partSizeInBytes;
    private final GenericMultipartHelper<PutObjectRequest, PutObjectResponse> genericMultipartHelper;
    private final long maxMemoryUsageInBytes;
    private final long multipartUploadThresholdInBytes;
    private final MultipartUploadHelper multipartUploadHelper;

    public UploadWithKnownContentLengthHelper(S3AsyncClient s3AsyncClient, long partSizeInBytes, long multipartUploadThresholdInBytes, long maxMemoryUsageInBytes) {
        this.s3AsyncClient = s3AsyncClient;
        this.partSizeInBytes = partSizeInBytes;
        this.genericMultipartHelper = new GenericMultipartHelper<PutObjectRequest, PutObjectResponse>(s3AsyncClient, SdkPojoConversionUtils::toAbortMultipartUploadRequest, SdkPojoConversionUtils::toPutObjectResponse);
        this.maxMemoryUsageInBytes = maxMemoryUsageInBytes;
        this.multipartUploadThresholdInBytes = multipartUploadThresholdInBytes;
        this.multipartUploadHelper = new MultipartUploadHelper(s3AsyncClient, multipartUploadThresholdInBytes, maxMemoryUsageInBytes);
    }

    public CompletableFuture<PutObjectResponse> uploadObject(PutObjectRequest putObjectRequest, AsyncRequestBody asyncRequestBody, long contentLength) {
        CompletableFuture<PutObjectResponse> returnFuture = new CompletableFuture<PutObjectResponse>();
        try {
            if (contentLength > this.multipartUploadThresholdInBytes && contentLength > this.partSizeInBytes) {
                log.debug(() -> "Starting the upload as multipart upload request");
                this.uploadInParts(putObjectRequest, contentLength, asyncRequestBody, returnFuture);
            } else {
                log.debug(() -> "Starting the upload as a single upload part request");
                this.multipartUploadHelper.uploadInOneChunk(putObjectRequest, asyncRequestBody, returnFuture);
            }
        }
        catch (Throwable throwable) {
            returnFuture.completeExceptionally(throwable);
        }
        return returnFuture;
    }

    private void uploadInParts(PutObjectRequest putObjectRequest, long contentLength, AsyncRequestBody asyncRequestBody, CompletableFuture<PutObjectResponse> returnFuture) {
        S3ResumeToken resumeToken = putObjectRequest.overrideConfiguration().map(c -> (S3ResumeToken)c.executionAttributes().getAttribute(S3MultipartExecutionAttribute.RESUME_TOKEN)).orElse(null);
        if (resumeToken == null) {
            this.initiateNewUpload(putObjectRequest, contentLength, asyncRequestBody, returnFuture);
        } else {
            ResumeRequestContext resumeRequestContext = new ResumeRequestContext(resumeToken, putObjectRequest, contentLength, asyncRequestBody, returnFuture);
            this.resumePausedUpload(resumeRequestContext);
        }
    }

    private void initiateNewUpload(PutObjectRequest putObjectRequest, long contentLength, AsyncRequestBody asyncRequestBody, CompletableFuture<PutObjectResponse> returnFuture) {
        CompletableFuture<CreateMultipartUploadResponse> createMultipartUploadFuture = this.multipartUploadHelper.createMultipartUpload(putObjectRequest, returnFuture);
        createMultipartUploadFuture.whenComplete((createMultipartUploadResponse, throwable) -> {
            if (throwable != null) {
                this.genericMultipartHelper.handleException(returnFuture, () -> "Failed to initiate multipart upload", (Throwable)throwable);
            } else {
                log.debug(() -> "Initiated a new multipart upload, uploadId: " + createMultipartUploadResponse.uploadId());
                this.uploadFromBeginning((Pair<PutObjectRequest, AsyncRequestBody>)Pair.of((Object)((Object)putObjectRequest), (Object)asyncRequestBody), contentLength, returnFuture, createMultipartUploadResponse.uploadId());
            }
        });
    }

    private void uploadFromBeginning(Pair<PutObjectRequest, AsyncRequestBody> request, long contentLength, CompletableFuture<PutObjectResponse> returnFuture, String uploadId) {
        long numPartsCompleted = 0L;
        long partSize = this.genericMultipartHelper.calculateOptimalPartSizeFor(contentLength, this.partSizeInBytes);
        int partCount = this.genericMultipartHelper.determinePartCount(contentLength, partSize);
        if (partSize > this.partSizeInBytes) {
            log.debug(() -> String.format("Configured partSize is %d, but using %d to prevent reaching maximum number of parts allowed", this.partSizeInBytes, partSize));
        }
        log.debug(() -> String.format("Starting multipart upload with partCount: %d, optimalPartSize: %d", partCount, partSize));
        MpuRequestContext mpuRequestContext = MpuRequestContext.builder().request(request).contentLength(contentLength).partSize(partSize).uploadId(uploadId).numPartsCompleted(numPartsCompleted).expectedNumParts(partCount).build();
        this.splitAndSubscribe(mpuRequestContext, returnFuture);
    }

    private void resumePausedUpload(ResumeRequestContext resumeContext) {
        S3ResumeToken resumeToken = resumeContext.resumeToken;
        String uploadId = resumeToken.uploadId();
        PutObjectRequest putObjectRequest = resumeContext.putObjectRequest;
        ConcurrentHashMap<Integer, CompletedPart> existingParts = new ConcurrentHashMap<Integer, CompletedPart>();
        CompletableFuture<Void> listPartsFuture = this.identifyExistingPartsForResume(uploadId, putObjectRequest, existingParts);
        int remainingParts = (int)(resumeToken.totalNumParts() - resumeToken.numPartsCompleted());
        log.debug(() -> String.format("Resuming a paused multipart upload, uploadId: %s, completedPartCount: %d, remainingPartCount: %d, partSize: %d", uploadId, resumeToken.numPartsCompleted(), remainingParts, resumeToken.partSize()));
        CompletableFutureUtils.forwardExceptionTo((CompletableFuture)resumeContext.returnFuture, listPartsFuture);
        listPartsFuture.whenComplete((r, t) -> {
            if (t != null) {
                this.genericMultipartHelper.handleException(resumeContext.returnFuture, () -> "Failed to resume because listParts failed", (Throwable)t);
                return;
            }
            Pair request = Pair.of((Object)((Object)putObjectRequest), (Object)resumeContext.asyncRequestBody);
            MpuRequestContext mpuRequestContext = MpuRequestContext.builder().request((Pair<PutObjectRequest, AsyncRequestBody>)request).contentLength(resumeContext.contentLength).partSize(resumeToken.partSize()).uploadId(uploadId).existingParts(existingParts).expectedNumParts(Math.toIntExact(resumeToken.totalNumParts())).numPartsCompleted(resumeToken.numPartsCompleted()).build();
            this.splitAndSubscribe(mpuRequestContext, resumeContext.returnFuture);
        });
    }

    private void splitAndSubscribe(MpuRequestContext mpuRequestContext, CompletableFuture<PutObjectResponse> returnFuture) {
        KnownContentLengthAsyncRequestBodySubscriber subscriber = new KnownContentLengthAsyncRequestBodySubscriber(mpuRequestContext, returnFuture, this.multipartUploadHelper);
        this.attachSubscriberToObservable(subscriber, (PutObjectRequest)((Object)mpuRequestContext.request().left()));
        ((AsyncRequestBody)mpuRequestContext.request().right()).split(b -> b.chunkSizeInBytes(mpuRequestContext.partSize()).bufferSizeInBytes(Long.valueOf(this.maxMemoryUsageInBytes))).subscribe((Subscriber)subscriber);
    }

    private CompletableFuture<Void> identifyExistingPartsForResume(String uploadId, PutObjectRequest putObjectRequest, Map<Integer, CompletedPart> existingParts) {
        ListPartsRequest request = SdkPojoConversionUtils.toListPartsRequest(uploadId, putObjectRequest);
        ListPartsPublisher listPartsPublisher = this.s3AsyncClient.listPartsPaginator(request);
        SdkPublisher<Part> partsPublisher = listPartsPublisher.parts();
        return partsPublisher.subscribe(part -> existingParts.put(part.partNumber(), SdkPojoConversionUtils.toCompletedPart(part)));
    }

    private void attachSubscriberToObservable(KnownContentLengthAsyncRequestBodySubscriber subscriber, PutObjectRequest putObjectRequest) {
        putObjectRequest.overrideConfiguration().map(c -> (PauseObservable)c.executionAttributes().getAttribute(S3MultipartExecutionAttribute.PAUSE_OBSERVABLE)).ifPresent(p -> p.setPausableUpload(new DefaultPausableUpload(subscriber)));
    }

    private static final class DefaultPausableUpload
    implements PausableUpload {
        private KnownContentLengthAsyncRequestBodySubscriber subscriber;

        private DefaultPausableUpload(KnownContentLengthAsyncRequestBodySubscriber subscriber) {
            this.subscriber = subscriber;
        }

        @Override
        public S3ResumeToken pause() {
            return this.subscriber.pause();
        }
    }

    private static final class ResumeRequestContext {
        private final S3ResumeToken resumeToken;
        private final PutObjectRequest putObjectRequest;
        private final long contentLength;
        private final AsyncRequestBody asyncRequestBody;
        private final CompletableFuture<PutObjectResponse> returnFuture;

        private ResumeRequestContext(S3ResumeToken resumeToken, PutObjectRequest putObjectRequest, long contentLength, AsyncRequestBody asyncRequestBody, CompletableFuture<PutObjectResponse> returnFuture) {
            this.resumeToken = resumeToken;
            this.putObjectRequest = putObjectRequest;
            this.contentLength = contentLength;
            this.asyncRequestBody = asyncRequestBody;
            this.returnFuture = returnFuture;
        }
    }
}

