/*
 * Decompiled with CFR 0.152.
 */
package jadx.plugins.input.javaconvert;

import jadx.api.plugins.JadxPluginContext;
import jadx.api.plugins.utils.CommonFileUtils;
import jadx.api.security.IJadxSecurity;
import jadx.plugins.input.javaconvert.AsmUtils;
import jadx.plugins.input.javaconvert.ConvertResult;
import jadx.plugins.input.javaconvert.D8Converter;
import jadx.plugins.input.javaconvert.DxConverter;
import jadx.plugins.input.javaconvert.JavaConvertOptions;
import jadx.zip.ZipReader;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.List;
import java.util.Objects;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JavaConvertLoader {
    private static final Logger LOG = LoggerFactory.getLogger(JavaConvertLoader.class);
    private final JavaConvertOptions options;
    private final ZipReader zipReader;
    private final IJadxSecurity security;

    public JavaConvertLoader(JavaConvertOptions options, JadxPluginContext context) {
        this.options = options;
        this.zipReader = context.getZipReader();
        this.security = context.getArgs().getSecurity();
    }

    public ConvertResult process(List<Path> input) {
        ConvertResult result = new ConvertResult();
        this.processJars(input, result);
        this.processAars(input, result);
        this.processClassFiles(input, result);
        return result;
    }

    private void processJars(List<Path> input, ConvertResult result) {
        PathMatcher jarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.jar");
        input.stream().filter(jarMatcher::matches).forEach(path -> {
            try {
                this.convertJar(result, (Path)path);
            }
            catch (Exception e15) {
                LOG.error("Failed to convert file: {}", (Object)path.toAbsolutePath(), (Object)e15);
            }
        });
    }

    private void processClassFiles(List<Path> input, ConvertResult result) {
        PathMatcher jarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.class");
        List clsFiles = input.stream().filter(jarMatcher::matches).collect(Collectors.toList());
        if (clsFiles.isEmpty()) {
            return;
        }
        try {
            LOG.debug("Converting class files ...");
            Path jarFile = Files.createTempFile("jadx-", ".jar", new FileAttribute[0]);
            try (JarOutputStream jo4 = new JarOutputStream(Files.newOutputStream(jarFile, new OpenOption[0]));){
                for (Path file2 : clsFiles) {
                    String clsName = AsmUtils.getNameFromClassFile(file2);
                    if (clsName == null) {
                        throw new IOException("Can't read class name from file: " + String.valueOf(file2));
                    }
                    if (!this.security.isValidEntryName(clsName)) {
                        LOG.warn("Skip class with invalid name: {}", (Object)clsName);
                        continue;
                    }
                    JavaConvertLoader.addFileToJar(jo4, file2, clsName + ".class");
                }
            }
            result.addTempPath(jarFile);
            LOG.debug("Packed {} class files into jar: {}", (Object)clsFiles.size(), (Object)jarFile);
            this.convertJar(result, jarFile);
        }
        catch (Exception e15) {
            LOG.error("Error process class files", e15);
        }
    }

    private void processAars(List<Path> input, ConvertResult result) {
        PathMatcher aarMatcher = FileSystems.getDefault().getPathMatcher("glob:**.aar");
        input.stream().filter(aarMatcher::matches).forEach(path -> this.zipReader.readEntries(path.toFile(), (entry, in4) -> {
            try {
                String entryName = entry.getName();
                if (entryName.endsWith(".jar")) {
                    Path tempJar = CommonFileUtils.saveToTempFile(in4, ".jar");
                    result.addTempPath(tempJar);
                    LOG.debug("Loading jar: {} ...", (Object)entryName);
                    this.convertJar(result, tempJar);
                }
            }
            catch (Exception e15) {
                LOG.error("Failed to process zip entry: {}", entry, (Object)e15);
            }
        }));
    }

    private void convertJar(ConvertResult result, Path path) throws Exception {
        if (this.repackAndConvertJar(result, path)) {
            return;
        }
        this.convertSimpleJar(result, path);
    }

    private boolean repackAndConvertJar(ConvertResult result, Path path) throws Exception {
        Boolean repackNeeded = this.zipReader.visitEntries(path.toFile(), zipEntry -> {
            String entryName = zipEntry.getName();
            if (zipEntry.isDirectory()) {
                if (entryName.equals("BOOT-INF/")) {
                    return true;
                }
                if (entryName.equals("META-INF/versions/")) {
                    return true;
                }
            }
            if (entryName.endsWith(".jar")) {
                return true;
            }
            if (entryName.endsWith("module-info.class")) {
                return true;
            }
            return null;
        });
        if (!Objects.equals(repackNeeded, Boolean.TRUE)) {
            return false;
        }
        LOG.debug("Repacking jar file: {} ...", (Object)path.toAbsolutePath());
        Path jarFile = Files.createTempFile("jadx-classes-", ".jar", new FileAttribute[0]);
        result.addTempPath(jarFile);
        try (JarOutputStream jo4 = new JarOutputStream(Files.newOutputStream(jarFile, new OpenOption[0]));){
            this.zipReader.readEntries(path.toFile(), (entry, in4) -> {
                try {
                    String entryName = entry.getName();
                    if (entryName.endsWith(".class")) {
                        if (entryName.endsWith("module-info.class") || entryName.startsWith("META-INF/versions/")) {
                            LOG.debug(" exclude: {}", (Object)entryName);
                            return;
                        }
                        byte[] clsFileContent = CommonFileUtils.loadBytes(in4);
                        String clsName = AsmUtils.getNameFromClassFile(clsFileContent);
                        if (clsName == null) {
                            throw new IOException("Can't read class name from file: " + entryName);
                        }
                        if (!this.security.isValidEntryName(clsName)) {
                            LOG.warn("Ignore class with invalid name: {} from {}", (Object)clsName, entry);
                        } else {
                            JavaConvertLoader.addJarEntry(jo4, clsName + ".class", clsFileContent, null);
                        }
                    } else if (entryName.endsWith(".jar")) {
                        Path tempJar = CommonFileUtils.saveToTempFile(in4, ".jar");
                        result.addTempPath(tempJar);
                        this.convertJar(result, tempJar);
                    }
                }
                catch (Exception e15) {
                    LOG.error("Failed to process jar entry: {} in {}", entry, path, e15);
                }
            });
        }
        this.convertSimpleJar(result, jarFile);
        return true;
    }

    private void convertSimpleJar(ConvertResult result, Path path) throws Exception {
        Path tempDirectory = Files.createTempDirectory("jadx-", new FileAttribute[0]);
        result.addTempPath(tempDirectory);
        LOG.debug("Converting to dex ...");
        this.convert(path, tempDirectory);
        List<Path> dexFiles = JavaConvertLoader.collectFilesInDir(tempDirectory);
        LOG.debug("Converted {} to {} dex", (Object)path.toAbsolutePath(), (Object)dexFiles.size());
        result.addConvertedFiles(dexFiles);
    }

    private void convert(Path path, Path tempDirectory) {
        JavaConvertOptions.Mode mode = this.options.getMode();
        switch (mode) {
            case DX: {
                try {
                    DxConverter.run(path, tempDirectory);
                }
                catch (Throwable e15) {
                    LOG.error("DX convert failed, path: {}", (Object)path, (Object)e15);
                }
                break;
            }
            case D8: {
                try {
                    D8Converter.run(path, tempDirectory, this.options);
                }
                catch (Throwable e16) {
                    LOG.error("D8 convert failed, path: {}", (Object)path, (Object)e16);
                }
                break;
            }
            case BOTH: {
                try {
                    DxConverter.run(path, tempDirectory);
                    break;
                }
                catch (Throwable e17) {
                    LOG.warn("DX convert failed, trying D8, path: {}", (Object)path);
                    try {
                        D8Converter.run(path, tempDirectory, this.options);
                        break;
                    }
                    catch (Throwable ex4) {
                        LOG.error("D8 convert failed: {}", (Object)ex4.getMessage());
                    }
                }
            }
        }
    }

    private static List<Path> collectFilesInDir(Path tempDirectory) throws IOException {
        PathMatcher dexMatcher = FileSystems.getDefault().getPathMatcher("glob:**.dex");
        try (Stream<Path> pathStream = Files.walk(tempDirectory, 1, new FileVisitOption[0]);){
            List<Path> list2 = pathStream.filter(p15 -> Files.isRegularFile(p15, LinkOption.NOFOLLOW_LINKS)).filter(dexMatcher::matches).collect(Collectors.toList());
            return list2;
        }
    }

    private static void addFileToJar(JarOutputStream jar, Path source, String entryName) throws IOException {
        byte[] fileContent = Files.readAllBytes(source);
        FileTime lastModifiedTime = Files.getLastModifiedTime(source, LinkOption.NOFOLLOW_LINKS);
        JavaConvertLoader.addJarEntry(jar, entryName, fileContent, lastModifiedTime);
    }

    private static void addJarEntry(JarOutputStream jar, String entryName, byte[] content, FileTime modTime) throws IOException {
        JarEntry entry = new JarEntry(entryName);
        if (modTime != null) {
            entry.setTime(modTime.toMillis());
        }
        jar.putNextEntry(entry);
        jar.write(content);
        jar.closeEntry();
    }
}

