﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/CapacityProviderField.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class DescribeCapacityProvidersRequest : public ECSRequest {
 public:
  AWS_ECS_API DescribeCapacityProvidersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeCapacityProviders"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of one or more capacity
   * providers. Up to <code>100</code> capacity providers can be described in an
   * action.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCapacityProviders() const { return m_capacityProviders; }
  inline bool CapacityProvidersHasBeenSet() const { return m_capacityProvidersHasBeenSet; }
  template <typename CapacityProvidersT = Aws::Vector<Aws::String>>
  void SetCapacityProviders(CapacityProvidersT&& value) {
    m_capacityProvidersHasBeenSet = true;
    m_capacityProviders = std::forward<CapacityProvidersT>(value);
  }
  template <typename CapacityProvidersT = Aws::Vector<Aws::String>>
  DescribeCapacityProvidersRequest& WithCapacityProviders(CapacityProvidersT&& value) {
    SetCapacityProviders(std::forward<CapacityProvidersT>(value));
    return *this;
  }
  template <typename CapacityProvidersT = Aws::String>
  DescribeCapacityProvidersRequest& AddCapacityProviders(CapacityProvidersT&& value) {
    m_capacityProvidersHasBeenSet = true;
    m_capacityProviders.emplace_back(std::forward<CapacityProvidersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the cluster to describe capacity providers for. When specified,
   * only capacity providers associated with this cluster are returned, including
   * Amazon ECS Managed Instances capacity providers.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  DescribeCapacityProvidersRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether or not you want to see the resource tags for the capacity
   * provider. If <code>TAGS</code> is specified, the tags are included in the
   * response. If this field is omitted, tags aren't included in the response.</p>
   */
  inline const Aws::Vector<CapacityProviderField>& GetInclude() const { return m_include; }
  inline bool IncludeHasBeenSet() const { return m_includeHasBeenSet; }
  template <typename IncludeT = Aws::Vector<CapacityProviderField>>
  void SetInclude(IncludeT&& value) {
    m_includeHasBeenSet = true;
    m_include = std::forward<IncludeT>(value);
  }
  template <typename IncludeT = Aws::Vector<CapacityProviderField>>
  DescribeCapacityProvidersRequest& WithInclude(IncludeT&& value) {
    SetInclude(std::forward<IncludeT>(value));
    return *this;
  }
  inline DescribeCapacityProvidersRequest& AddInclude(CapacityProviderField value) {
    m_includeHasBeenSet = true;
    m_include.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of account setting results returned by
   * <code>DescribeCapacityProviders</code> in paginated output. When this parameter
   * is used, <code>DescribeCapacityProviders</code> only returns
   * <code>maxResults</code> results in a single page along with a
   * <code>nextToken</code> response element. The remaining results of the initial
   * request can be seen by sending another <code>DescribeCapacityProviders</code>
   * request with the returned <code>nextToken</code> value. This value can be
   * between 1 and 10. If this parameter is not used, then
   * <code>DescribeCapacityProviders</code> returns up to 10 results and a
   * <code>nextToken</code> value if applicable.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeCapacityProvidersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated
   * <code>DescribeCapacityProviders</code> request where <code>maxResults</code> was
   * used and the results exceeded the value of that parameter. Pagination continues
   * from the end of the previous results that returned the <code>nextToken</code>
   * value.</p>  <p>This token should be treated as an opaque identifier that
   * is only used to retrieve the next items in a list and not for other programmatic
   * purposes.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeCapacityProvidersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_capacityProviders;

  Aws::String m_cluster;

  Aws::Vector<CapacityProviderField> m_include;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_capacityProvidersHasBeenSet = false;
  bool m_clusterHasBeenSet = false;
  bool m_includeHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
