﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/securityhub/model/VulnerabilityVendor.h>
#include <aws/securityhub/model/VulnerabilityFixAvailable.h>
#include <aws/securityhub/model/VulnerabilityExploitAvailable.h>
#include <aws/securityhub/model/SoftwarePackage.h>
#include <aws/securityhub/model/Cvss.h>
#include <aws/securityhub/model/VulnerabilityCodeVulnerabilities.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace SecurityHub
{
namespace Model
{

  /**
   * <p>A vulnerability associated with a finding.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/Vulnerability">AWS
   * API Reference</a></p>
   */
  class Vulnerability
  {
  public:
    AWS_SECURITYHUB_API Vulnerability() = default;
    AWS_SECURITYHUB_API Vulnerability(Aws::Utils::Json::JsonView jsonValue);
    AWS_SECURITYHUB_API Vulnerability& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_SECURITYHUB_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>The identifier of the vulnerability.</p>
     */
    inline const Aws::String& GetId() const { return m_id; }
    inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
    template<typename IdT = Aws::String>
    void SetId(IdT&& value) { m_idHasBeenSet = true; m_id = std::forward<IdT>(value); }
    template<typename IdT = Aws::String>
    Vulnerability& WithId(IdT&& value) { SetId(std::forward<IdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>List of software packages that have the vulnerability.</p>
     */
    inline const Aws::Vector<SoftwarePackage>& GetVulnerablePackages() const { return m_vulnerablePackages; }
    inline bool VulnerablePackagesHasBeenSet() const { return m_vulnerablePackagesHasBeenSet; }
    template<typename VulnerablePackagesT = Aws::Vector<SoftwarePackage>>
    void SetVulnerablePackages(VulnerablePackagesT&& value) { m_vulnerablePackagesHasBeenSet = true; m_vulnerablePackages = std::forward<VulnerablePackagesT>(value); }
    template<typename VulnerablePackagesT = Aws::Vector<SoftwarePackage>>
    Vulnerability& WithVulnerablePackages(VulnerablePackagesT&& value) { SetVulnerablePackages(std::forward<VulnerablePackagesT>(value)); return *this;}
    template<typename VulnerablePackagesT = SoftwarePackage>
    Vulnerability& AddVulnerablePackages(VulnerablePackagesT&& value) { m_vulnerablePackagesHasBeenSet = true; m_vulnerablePackages.emplace_back(std::forward<VulnerablePackagesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>CVSS scores from the advisory related to the vulnerability.</p>
     */
    inline const Aws::Vector<Cvss>& GetCvss() const { return m_cvss; }
    inline bool CvssHasBeenSet() const { return m_cvssHasBeenSet; }
    template<typename CvssT = Aws::Vector<Cvss>>
    void SetCvss(CvssT&& value) { m_cvssHasBeenSet = true; m_cvss = std::forward<CvssT>(value); }
    template<typename CvssT = Aws::Vector<Cvss>>
    Vulnerability& WithCvss(CvssT&& value) { SetCvss(std::forward<CvssT>(value)); return *this;}
    template<typename CvssT = Cvss>
    Vulnerability& AddCvss(CvssT&& value) { m_cvssHasBeenSet = true; m_cvss.emplace_back(std::forward<CvssT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>List of vulnerabilities that are related to this vulnerability.</p>
     */
    inline const Aws::Vector<Aws::String>& GetRelatedVulnerabilities() const { return m_relatedVulnerabilities; }
    inline bool RelatedVulnerabilitiesHasBeenSet() const { return m_relatedVulnerabilitiesHasBeenSet; }
    template<typename RelatedVulnerabilitiesT = Aws::Vector<Aws::String>>
    void SetRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { m_relatedVulnerabilitiesHasBeenSet = true; m_relatedVulnerabilities = std::forward<RelatedVulnerabilitiesT>(value); }
    template<typename RelatedVulnerabilitiesT = Aws::Vector<Aws::String>>
    Vulnerability& WithRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { SetRelatedVulnerabilities(std::forward<RelatedVulnerabilitiesT>(value)); return *this;}
    template<typename RelatedVulnerabilitiesT = Aws::String>
    Vulnerability& AddRelatedVulnerabilities(RelatedVulnerabilitiesT&& value) { m_relatedVulnerabilitiesHasBeenSet = true; m_relatedVulnerabilities.emplace_back(std::forward<RelatedVulnerabilitiesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Information about the vendor that generates the vulnerability report.</p>
     */
    inline const VulnerabilityVendor& GetVendor() const { return m_vendor; }
    inline bool VendorHasBeenSet() const { return m_vendorHasBeenSet; }
    template<typename VendorT = VulnerabilityVendor>
    void SetVendor(VendorT&& value) { m_vendorHasBeenSet = true; m_vendor = std::forward<VendorT>(value); }
    template<typename VendorT = VulnerabilityVendor>
    Vulnerability& WithVendor(VendorT&& value) { SetVendor(std::forward<VendorT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>A list of URLs that provide additional information about the
     * vulnerability.</p>
     */
    inline const Aws::Vector<Aws::String>& GetReferenceUrls() const { return m_referenceUrls; }
    inline bool ReferenceUrlsHasBeenSet() const { return m_referenceUrlsHasBeenSet; }
    template<typename ReferenceUrlsT = Aws::Vector<Aws::String>>
    void SetReferenceUrls(ReferenceUrlsT&& value) { m_referenceUrlsHasBeenSet = true; m_referenceUrls = std::forward<ReferenceUrlsT>(value); }
    template<typename ReferenceUrlsT = Aws::Vector<Aws::String>>
    Vulnerability& WithReferenceUrls(ReferenceUrlsT&& value) { SetReferenceUrls(std::forward<ReferenceUrlsT>(value)); return *this;}
    template<typename ReferenceUrlsT = Aws::String>
    Vulnerability& AddReferenceUrls(ReferenceUrlsT&& value) { m_referenceUrlsHasBeenSet = true; m_referenceUrls.emplace_back(std::forward<ReferenceUrlsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Specifies if all vulnerable packages in a finding have a value for
     * <code>FixedInVersion</code> and <code>Remediation</code>. This field is
     * evaluated for each vulnerability <code>Id</code> based on the number of
     * vulnerable packages that have a value for both <code>FixedInVersion</code> and
     * <code>Remediation</code>. Valid values are as follows:</p> <ul> <li> <p>
     * <code>YES</code> if all vulnerable packages have a value for both
     * <code>FixedInVersion</code> and <code>Remediation</code> </p> </li> <li> <p>
     * <code>NO</code> if no vulnerable packages have a value for
     * <code>FixedInVersion</code> and <code>Remediation</code> </p> </li> <li> <p>
     * <code>PARTIAL</code> otherwise</p> </li> </ul>
     */
    inline VulnerabilityFixAvailable GetFixAvailable() const { return m_fixAvailable; }
    inline bool FixAvailableHasBeenSet() const { return m_fixAvailableHasBeenSet; }
    inline void SetFixAvailable(VulnerabilityFixAvailable value) { m_fixAvailableHasBeenSet = true; m_fixAvailable = value; }
    inline Vulnerability& WithFixAvailable(VulnerabilityFixAvailable value) { SetFixAvailable(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The Exploit Prediction Scoring System (EPSS) score for a finding. </p>
     */
    inline double GetEpssScore() const { return m_epssScore; }
    inline bool EpssScoreHasBeenSet() const { return m_epssScoreHasBeenSet; }
    inline void SetEpssScore(double value) { m_epssScoreHasBeenSet = true; m_epssScore = value; }
    inline Vulnerability& WithEpssScore(double value) { SetEpssScore(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Whether an exploit is available for a finding. </p>
     */
    inline VulnerabilityExploitAvailable GetExploitAvailable() const { return m_exploitAvailable; }
    inline bool ExploitAvailableHasBeenSet() const { return m_exploitAvailableHasBeenSet; }
    inline void SetExploitAvailable(VulnerabilityExploitAvailable value) { m_exploitAvailableHasBeenSet = true; m_exploitAvailable = value; }
    inline Vulnerability& WithExploitAvailable(VulnerabilityExploitAvailable value) { SetExploitAvailable(value); return *this;}
    ///@}

    ///@{
    /**
     * <p> The date and time of the last exploit associated with a finding discovered
     * in your environment. </p>
     */
    inline const Aws::String& GetLastKnownExploitAt() const { return m_lastKnownExploitAt; }
    inline bool LastKnownExploitAtHasBeenSet() const { return m_lastKnownExploitAtHasBeenSet; }
    template<typename LastKnownExploitAtT = Aws::String>
    void SetLastKnownExploitAt(LastKnownExploitAtT&& value) { m_lastKnownExploitAtHasBeenSet = true; m_lastKnownExploitAt = std::forward<LastKnownExploitAtT>(value); }
    template<typename LastKnownExploitAtT = Aws::String>
    Vulnerability& WithLastKnownExploitAt(LastKnownExploitAtT&& value) { SetLastKnownExploitAt(std::forward<LastKnownExploitAtT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The vulnerabilities found in your Lambda function code. This field pertains
     * to findings that Security Hub receives from Amazon Inspector. </p>
     */
    inline const Aws::Vector<VulnerabilityCodeVulnerabilities>& GetCodeVulnerabilities() const { return m_codeVulnerabilities; }
    inline bool CodeVulnerabilitiesHasBeenSet() const { return m_codeVulnerabilitiesHasBeenSet; }
    template<typename CodeVulnerabilitiesT = Aws::Vector<VulnerabilityCodeVulnerabilities>>
    void SetCodeVulnerabilities(CodeVulnerabilitiesT&& value) { m_codeVulnerabilitiesHasBeenSet = true; m_codeVulnerabilities = std::forward<CodeVulnerabilitiesT>(value); }
    template<typename CodeVulnerabilitiesT = Aws::Vector<VulnerabilityCodeVulnerabilities>>
    Vulnerability& WithCodeVulnerabilities(CodeVulnerabilitiesT&& value) { SetCodeVulnerabilities(std::forward<CodeVulnerabilitiesT>(value)); return *this;}
    template<typename CodeVulnerabilitiesT = VulnerabilityCodeVulnerabilities>
    Vulnerability& AddCodeVulnerabilities(CodeVulnerabilitiesT&& value) { m_codeVulnerabilitiesHasBeenSet = true; m_codeVulnerabilities.emplace_back(std::forward<CodeVulnerabilitiesT>(value)); return *this; }
    ///@}
  private:

    Aws::String m_id;
    bool m_idHasBeenSet = false;

    Aws::Vector<SoftwarePackage> m_vulnerablePackages;
    bool m_vulnerablePackagesHasBeenSet = false;

    Aws::Vector<Cvss> m_cvss;
    bool m_cvssHasBeenSet = false;

    Aws::Vector<Aws::String> m_relatedVulnerabilities;
    bool m_relatedVulnerabilitiesHasBeenSet = false;

    VulnerabilityVendor m_vendor;
    bool m_vendorHasBeenSet = false;

    Aws::Vector<Aws::String> m_referenceUrls;
    bool m_referenceUrlsHasBeenSet = false;

    VulnerabilityFixAvailable m_fixAvailable{VulnerabilityFixAvailable::NOT_SET};
    bool m_fixAvailableHasBeenSet = false;

    double m_epssScore{0.0};
    bool m_epssScoreHasBeenSet = false;

    VulnerabilityExploitAvailable m_exploitAvailable{VulnerabilityExploitAvailable::NOT_SET};
    bool m_exploitAvailableHasBeenSet = false;

    Aws::String m_lastKnownExploitAt;
    bool m_lastKnownExploitAtHasBeenSet = false;

    Aws::Vector<VulnerabilityCodeVulnerabilities> m_codeVulnerabilities;
    bool m_codeVulnerabilitiesHasBeenSet = false;
  };

} // namespace Model
} // namespace SecurityHub
} // namespace Aws
