/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.plantuml.code.deflate;

import java.io.EOFException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.zip.DataFormatException;
import net.sourceforge.plantuml.code.deflate.BitInputStream;
import net.sourceforge.plantuml.code.deflate.ByteHistory;
import net.sourceforge.plantuml.code.deflate.CanonicalCode;
import net.sourceforge.plantuml.code.deflate.OutputStreamProtected;

public final class Decompressor {
    private BitInputStream input;
    private OutputStreamProtected output;
    private ByteHistory dictionary;
    private static final CanonicalCode FIXED_LITERAL_LENGTH_CODE;
    private static final CanonicalCode FIXED_DISTANCE_CODE;

    public static byte[] decompress(BitInputStream in) throws IOException, DataFormatException {
        OutputStreamProtected out = new OutputStreamProtected();
        Decompressor.decompress(in, out);
        return out.toByteArray();
    }

    public static void decompress(BitInputStream in, OutputStreamProtected out) throws IOException, DataFormatException {
        new Decompressor(in, out);
    }

    private Decompressor(BitInputStream in, OutputStreamProtected out) throws IOException, DataFormatException {
        boolean isFinal;
        this.input = Objects.requireNonNull(in);
        this.output = Objects.requireNonNull(out);
        this.dictionary = new ByteHistory(32768);
        do {
            isFinal = in.readNoEof() == 1;
            int type = this.readInt(2);
            if (type == 0) {
                this.decompressUncompressedBlock();
                continue;
            }
            if (type == 1) {
                this.decompressHuffmanBlock(FIXED_LITERAL_LENGTH_CODE, FIXED_DISTANCE_CODE);
                continue;
            }
            if (type == 2) {
                CanonicalCode[] litLenAndDist = this.decodeHuffmanCodes();
                this.decompressHuffmanBlock(litLenAndDist[0], litLenAndDist[1]);
                continue;
            }
            if (type == 3) {
                throw new DataFormatException("Reserved block type");
            }
            throw new IllegalStateException("Impossible value");
        } while (!isFinal);
    }

    private CanonicalCode[] decodeHuffmanCodes() throws IOException, DataFormatException {
        CanonicalCode distCode;
        CanonicalCode litLenCode;
        CanonicalCode codeLenCode;
        int numLitLenCodes = this.readInt(5) + 257;
        int numDistCodes = this.readInt(5) + 1;
        int numCodeLenCodes = this.readInt(4) + 4;
        int[] codeLenCodeLen = new int[19];
        codeLenCodeLen[16] = this.readInt(3);
        codeLenCodeLen[17] = this.readInt(3);
        codeLenCodeLen[18] = this.readInt(3);
        codeLenCodeLen[0] = this.readInt(3);
        for (int i = 0; i < numCodeLenCodes - 4; ++i) {
            int j = i % 2 == 0 ? 8 + i / 2 : 7 - i / 2;
            codeLenCodeLen[j] = this.readInt(3);
        }
        try {
            codeLenCode = new CanonicalCode(codeLenCodeLen);
        }
        catch (IllegalArgumentException e) {
            throw new DataFormatException(e.getMessage());
        }
        int[] codeLens = new int[numLitLenCodes + numDistCodes];
        int codeLensIndex = 0;
        while (codeLensIndex < codeLens.length) {
            int runLen;
            int sym = codeLenCode.decodeNextSymbol(this.input);
            if (0 <= sym && sym <= 15) {
                codeLens[codeLensIndex] = sym;
                ++codeLensIndex;
                continue;
            }
            int runVal = 0;
            if (sym == 16) {
                if (codeLensIndex == 0) {
                    throw new DataFormatException("No code length value to copy");
                }
                runLen = this.readInt(2) + 3;
                runVal = codeLens[codeLensIndex - 1];
            } else if (sym == 17) {
                runLen = this.readInt(3) + 3;
            } else if (sym == 18) {
                runLen = this.readInt(7) + 11;
            } else {
                throw new IllegalStateException("Symbol out of range");
            }
            int end = codeLensIndex + runLen;
            if (end > codeLens.length) {
                throw new DataFormatException("Run exceeds number of codes");
            }
            Arrays.fill(codeLens, codeLensIndex, end, runVal);
            codeLensIndex = end;
        }
        int[] litLenCodeLen = Arrays.copyOf(codeLens, numLitLenCodes);
        try {
            litLenCode = new CanonicalCode(litLenCodeLen);
        }
        catch (IllegalArgumentException e) {
            throw new DataFormatException(e.getMessage());
        }
        int[] distCodeLen = Arrays.copyOfRange(codeLens, numLitLenCodes, codeLens.length);
        if (distCodeLen.length == 1 && distCodeLen[0] == 0) {
            distCode = null;
        } else {
            int oneCount = 0;
            int otherPositiveCount = 0;
            for (int x : distCodeLen) {
                if (x == 1) {
                    ++oneCount;
                    continue;
                }
                if (x <= 1) continue;
                ++otherPositiveCount;
            }
            if (oneCount == 1 && otherPositiveCount == 0) {
                distCodeLen = Arrays.copyOf(distCodeLen, 32);
                distCodeLen[31] = 1;
            }
            try {
                distCode = new CanonicalCode(distCodeLen);
            }
            catch (IllegalArgumentException e) {
                throw new DataFormatException(e.getMessage());
            }
        }
        return new CanonicalCode[]{litLenCode, distCode};
    }

    private void decompressUncompressedBlock() throws IOException, DataFormatException {
        int nlen;
        while (this.input.getBitPosition() != 0) {
            this.input.readNoEof();
        }
        int len = this.readInt(16);
        if ((len ^ 0xFFFF) != (nlen = this.readInt(16))) {
            throw new DataFormatException("Invalid length in uncompressed block");
        }
        for (int i = 0; i < len; ++i) {
            int b = this.input.readByte();
            if (b == -1) {
                throw new EOFException();
            }
            this.output.write(b);
            this.dictionary.append(b);
        }
    }

    private void decompressHuffmanBlock(CanonicalCode litLenCode, CanonicalCode distCode) throws IOException, DataFormatException {
        int sym;
        Objects.requireNonNull(litLenCode);
        while ((sym = litLenCode.decodeNextSymbol(this.input)) != 256) {
            if (sym < 256) {
                this.output.write(sym);
                this.dictionary.append(sym);
                continue;
            }
            int run = this.decodeRunLength(sym);
            if (run < 3 || run > 258) {
                throw new IllegalStateException("Invalid run length");
            }
            if (distCode == null) {
                throw new DataFormatException("Length symbol encountered with empty distance code");
            }
            int distSym = distCode.decodeNextSymbol(this.input);
            int dist = this.decodeDistance(distSym);
            if (dist < 1 || dist > 32768) {
                throw new IllegalStateException("Invalid distance");
            }
            this.dictionary.copy(dist, run, this.output);
        }
    }

    private int decodeRunLength(int sym) throws IOException, DataFormatException {
        if (sym < 257 || sym > 287) {
            throw new IllegalStateException("Invalid run length symbol: " + sym);
        }
        if (sym <= 264) {
            return sym - 254;
        }
        if (sym <= 284) {
            int numExtraBits = (sym - 261) / 4;
            return ((sym - 265) % 4 + 4 << numExtraBits) + 3 + this.readInt(numExtraBits);
        }
        if (sym == 285) {
            return 258;
        }
        throw new DataFormatException("Reserved length symbol: " + sym);
    }

    private int decodeDistance(int sym) throws IOException, DataFormatException {
        if (sym < 0 || sym > 31) {
            throw new IllegalStateException("Invalid distance symbol: " + sym);
        }
        if (sym <= 3) {
            return sym + 1;
        }
        if (sym <= 29) {
            int numExtraBits = sym / 2 - 1;
            return (sym % 2 + 2 << numExtraBits) + 1 + this.readInt(numExtraBits);
        }
        throw new DataFormatException("Reserved distance symbol: " + sym);
    }

    private int readInt(int numBits) throws IOException {
        if (numBits < 0 || numBits > 31) {
            throw new IllegalArgumentException();
        }
        int result = 0;
        for (int i = 0; i < numBits; ++i) {
            result |= this.input.readNoEof() << i;
        }
        return result;
    }

    static {
        int[] llcodelens = new int[288];
        Arrays.fill(llcodelens, 0, 144, 8);
        Arrays.fill(llcodelens, 144, 256, 9);
        Arrays.fill(llcodelens, 256, 280, 7);
        Arrays.fill(llcodelens, 280, 288, 8);
        FIXED_LITERAL_LENGTH_CODE = new CanonicalCode(llcodelens);
        int[] distcodelens = new int[32];
        Arrays.fill(distcodelens, 5);
        FIXED_DISTANCE_CODE = new CanonicalCode(distcodelens);
    }
}

